/**********************************************************************
// Copyright 2018 Autodesk, Inc.  All rights reserved. 
**********************************************************************/
-- Provides the dialogs for the Security Tools
-- version 1.0 - initial version of script in package file
-- version 1.0.1 - support 3ds Max 2016 and later, set up to use help file from application bundle package

-- define a temporary version of SecurityToolsDialogs_structdef so that we can use the member variable as part of the rollout definitions in the real version
-- of SecurityToolsDialogs_structdef.
struct SecurityToolsDialogs_structdef 
(
	m_filelist = #()
)
::SecurityToolsDialogs_instance = SecurityToolsDialogs_structdef()

struct SecurityToolsDialogs_structdef 
(
	private
	-- icon file names. Includes whether light/dark theme and size. Set by create handler.
	m_error_icon_filename,
	m_info_icon_filename,
	m_success_icon_filename,
	m_warning_icon_filename,
	m_help_filename,

	-- The dialog background color. Set by create handler. Used when creating the dialogs and for alpha compositing of icons.
	m_dialog_background_color,

	-- whether to by default use the embedded localized help files from the application bundle package (true) or the help topics from the 3ds Max 
	m_help_use_bundle_help = true,
	
	-- the help topic id for the 'Learn about corruption' and 'Learn more' buttons and F1 handling, Filled in by method displaying dialog.
	-- used when not using embedded localized help file
	m_help_topic_id,
	
	-- the leaf file name of the embedded help file in the application bundle package. File will be in 3dsMaxSecurityTools\Contents\Help\<lang>
	m_help_bundle_filename = "index.htm",
	
	public
	-- holds file list to display in some dialogs, Filled in by method displaying dialog.
	m_filelist = #(),
	
	-- this works around maxscript issues with using bitmaps containing alpha channel as imgtag bitmaps.
	fn set_imgtag_bitmap theImgTag theBitmap_filename =
	(
		-- alpha blend the icon image with a bitmap of same color as dialog background.
		local theBitmap = openBitMap theBitmap_filename gamma:1
		if theBitmap != undefined do
		(
			local result_bitmap = bitmap theBitmap.width theBitmap.height color:m_dialog_background_color gamma:1.
			pasteBitmap theBitmap result_bitmap [0,0] [0,0] type:#blend 
			-- need to turn off display gamma when setting the imgtag bitmap
			local old_colorCorrectionMode = iDisplayGamma.colorCorrectionMode
			iDisplayGamma.colorCorrectionMode = #none
			theImgTag.bitmap = result_bitmap
			iDisplayGamma.colorCorrectionMode = old_colorCorrectionMode
		)
		ok
	),
	
	-- displays the help file if using bundled help file, or the 3ds Max Help file for a topic id
	-- if topic_id is undefined, display bundled help file.
	fn show_help_for_topic topic_id = 
	(
		if m_help_use_bundle_help or topic_id == undefined then
		(
			if m_help_filename != undefined and securitytools != undefined do
				securitytools.ShowProductHelpForTopic m_help_filename
		)
		else
		(
			if securitytools != undefined do
				securitytools.ShowProductHelpForTopic topic_id
		)
	),
	
	-- the display_corruptionFoundInScene_dialog function return value
	m_display_corruptionFoundInScene_dialog_return_value,
	
	 -- the (Autodesk.Max.MaxPlus.Application).EnableDrawing method. Not directly exposed via mxs, so acquired from MaxPlus
	m_enableDrawing_method,
	
	-- the name of the 3rd party script found
	m_which_script,
	
	-- the rollout definition for corruption found in scene
	m_corruptionFoundInScene_ro = rollout m_corruptionFoundInScene_ro ~CORRUPTION_FOUND_IN_SCENE_TITLE~ width:~CORRUPTION_FOUND_IN_SCENE_WIDTH~
	(
		local autobackup_enabled -- whether autobackup was enabled when opening the dialog
		local enableDrawing_method -- the (Autodesk.Max.MaxPlus.Application).EnableDrawing method. Not directly exposed via mxs, so acquired from MaxPlus
		imgtag icon width:32 height:32 align:#left style:#bmp_center applyUIScaling:false
		label l1 ~PROTECT_YOUR_WORK~ offset:~MAIN_INSTRUCTION_OFFSET~ align:#left
		label l2 ""
		label l3 ~SCENE_IS_CORRUPT~  align:#left height:~SCENE_IS_CORRUPT_HEIGHT~
		label l4 ~RECOMMEND_SCENE_CLEAN~ align:#left
		label l5 ""
		button b_learn ~LEARN~ across:3 align:#left
		button b_clean ~CLEAN~ align:#right offset:~CLEAN_OFFSET_SCENE~
		button b_leave ~LEAVE~ align:#right
		
		on m_corruptionFoundInScene_ro open do
		(
			set_imgtag_bitmap icon m_warning_icon_filename
			l3.text = substituteString l3.text "%" m_which_script
			m_display_corruptionFoundInScene_dialog_return_value = false -- default return value
			autobackup_enabled = autobackup.enabled  -- disable autobackup while dialog is displayed
			autobackup.enabled = false
			-- We want to disable viewport redraws while this dialog is up, otherwise contaminated scripted controllers will be evaluated in the viewport redraws
			-- while this dialog is displayed, before we have a chance to clean the controllers.
			-- We really only need to do this when called from test_for_ALC_scripted_controller_callback(), as viewport redraw is already disabled during normal scene load and merge.
			-- Note that the EnableDrawing method needs to be called in pairs of disabled/enabled. Implementation is 'disabled depth', where redraw is enabled in disabled count is 0.
			if m_enableDrawing_method != undefined do -- disable viewport redraw while dialog is displayed, otherwise redraw may occur causing scripted controllers to be evaluated
				m_enableDrawing_method false
		)
		on m_corruptionFoundInScene_ro close do
		(
			autobackup.enabled = autobackup_enabled
			if m_enableDrawing_method != undefined do
				m_enableDrawing_method true
		)
		on b_learn pressed do
		(
			show_help_for_topic m_help_topic_id
		)
		on b_clean pressed do
		(
			m_display_corruptionFoundInScene_dialog_return_value = true
			destroyDialog m_corruptionFoundInScene_ro
		)
		on b_leave pressed do
		(
			m_display_corruptionFoundInScene_dialog_return_value = false
			destroyDialog m_corruptionFoundInScene_ro
		)
		on m_corruptionFoundInScene_ro help do
		(
			show_help_for_topic m_help_topic_id
		)
	),

	-- display corruption found in scene dialog. Returns whether to clean the scene.
	fn display_corruptionFoundInScene_dialog help_topic_id which_script =
	(
		m_help_topic_id = help_topic_id
		m_which_script = which_script
		createdialog m_corruptionFoundInScene_ro bgcolor:m_dialog_background_color modal:true escapeEnable:true
		m_display_corruptionFoundInScene_dialog_return_value -- return value
	),

	-- the rollout definition for corruption cleaned in scene
	m_corruptionCleanedInScene_ro = rollout m_corruptionCleanedInScene_ro ~CORRUPTION_CLEANED_TITLE~ width:~CORRUPTION_CLEANED_WIDTH~
	(
		local autobackup_enabled -- whether autobackup was enabled when opening the dialog
		imgtag icon width:32 height:32 align:#left style:#bmp_center applyUIScaling:false
		label l1 ~NEXT_STEPS~ offset:~MAIN_INSTRUCTION_OFFSET~ align:#left
		label l2 ""
		label l3 ~SAVE_SCENE~ align:#left
		label l4 ""
		button b_close ~CLOSE~ align:#right
		
		on m_corruptionCleanedInScene_ro open do
		(
			set_imgtag_bitmap icon m_info_icon_filename
			autobackup_enabled = autobackup.enabled  -- disable autobackup while dialog is displayed
			autobackup.enabled = false
		)
		on m_corruptionCleanedInScene_ro close do
		(
			autobackup.enabled = autobackup_enabled
		)
		on b_close pressed do
		(
			destroyDialog m_corruptionCleanedInScene_ro
		)
		on m_corruptionCleanedInScene_ro help do
		(
			show_help_for_topic m_help_topic_id
		)
	),

	-- display corruption cleaned in scene dialog
	fn display_corruptionCleanedInScene_dialog help_topic_id =
	(
		m_help_topic_id = help_topic_id
		createdialog m_corruptionCleanedInScene_ro bgcolor:m_dialog_background_color modal:true escapeEnable:true
	),

	-- the rollout definition for corruption not cleaned in scene
	m_corruptionNotCleanedInScene_ro = rollout m_corruptionNotCleanedInScene_ro ~SCENE_CORRUPTION_NOT_CLEANED_TITLE~ width:~SCENE_CORRUPTION_NOT_CLEANED_WIDTH~
	(
		local autobackup_enabled -- whether autobackup was enabled when opening the dialog
		imgtag icon width:32 height:32 align:#left style:#bmp_center applyUIScaling:false
		label l1 ~SCENE_CORRUPT_SCRIPTS_NOT_CLEANED~ offset:~MAIN_INSTRUCTION_OFFSET~ align:#left
		label l2 ""
		label l3 ~SCENE_CORRUPTION_NOT_CLEANED_MSG~ height:~SCENE_CORRUPTION_NOT_CLEANED_MSG_HEIGHT~ align:#left
		label l4 ""
		button b_learn ~LEARN~ across:2 align:#left
		button b_close ~CLOSE~ align:#right
		
		on m_corruptionNotCleanedInScene_ro open do
		(
			set_imgtag_bitmap icon m_warning_icon_filename
			autobackup_enabled = autobackup.enabled  -- disable autobackup while dialog is displayed
			autobackup.enabled = false
		)
		on m_corruptionNotCleanedInScene_ro close do
		(
			autobackup.enabled = autobackup_enabled
		)
		on b_learn pressed do
		(
			show_help_for_topic m_help_topic_id
		)
		on b_close pressed do
		(
			destroyDialog m_corruptionNotCleanedInScene_ro
		)
		on m_corruptionNotCleanedInScene_ro help do
		(
			show_help_for_topic m_help_topic_id
		)
	),

	-- display corruption not cleaned in scene dialog
	fn display_corruptionNotCleanedInScene_dialog help_topic_id =
	(
		m_help_topic_id = help_topic_id
		createdialog m_corruptionNotCleanedInScene_ro bgcolor:m_dialog_background_color modal:true escapeEnable:true
	),

	-- the display_corruptionFoundInEnv_dialog function return value
	m_display_corruptionFoundInEnv_dialog_return_value,
	
	-- the filelist count for the first time the m_corruptionFoundInEnv_ro is displayed. We are playing games with the et_filelist height and the layout of controls under it. Need
	-- this value to tweak the rollout size
	m_corruptionFoundInEnv_ro_first_filelist_count,

	-- the rollout definition for corruption found in env
	m_corruptionFoundInEnv_ro = rollout m_corruptionFoundInEnv_ro ~CORRUPTION_FOUND_IN_ENV_TITLE~ width:~CORRUPTION_FOUND_IN_ENV_WIDTH~
	(
		-- local initial values are evaluated at compile time. We use m_filelist_height as the height for et_filelist so that controls under it are laid out correctly and the rollout height is ok.
		local m_filelist_height = if SecurityToolsDialogs_instance.m_filelist == undefined then 16 else SecurityToolsDialogs_instance.m_filelist.count * 16
		local autobackup_enabled -- whether autobackup was enabled when opening the dialog
		local m_orig_height -- restore initial rollout height in close handler
			
		imgtag icon width:32 height:32 align:#left style:#bmp_center applyUIScaling:false
		label l1 ~PROTECT_YOUR_WORK~ offset:~MAIN_INSTRUCTION_OFFSET~ align:#left
		label l2 ""
		label l3 ~ENV_IS_CORRUPT~ align:#left height:~ENV_IS_CORRUPT_HEIGHT~
		label l4 ~RECOMMEND_ENV_CLEAN~ align:#left
		edittext et_filelist height:m_filelist_height readOnly:true
		label l5 ""
		button b_learn ~LEARN~ across:3 align:#left
		button b_clean ~CLEAN~ align:#right offset:~CLEAN_OFFSET_ENV~
		button b_leave ~LEAVE~ align:#right
		
		on m_corruptionFoundInEnv_ro open do
		(
			set_imgtag_bitmap icon m_warning_icon_filename
			l3.text = substituteString l3.text "%" m_which_script
			local content = ""
			local first_entry = true
			for filename in m_filelist do 
			(
				if not first_entry do content += "\n"
				first_entry = false
				content += filename
			)
			et_filelist.text = content
			
			-- futz with height of rollout. May be easier to not use m_filelist_height as param for et_filelist, but rather set the height in this open handler
			-- and manually reposition controls below it.
			m_orig_height = m_corruptionFoundInEnv_ro.height
			if m_corruptionFoundInEnv_ro_first_filelist_count == undefined then
				m_corruptionFoundInEnv_ro_first_filelist_count = m_filelist.count
			m_corruptionFoundInEnv_ro.height += (m_filelist.count - m_corruptionFoundInEnv_ro_first_filelist_count) * 16 
			-- for 3ds Max 2016 and earlier following is needed to get the dialog height to update when inside the open handler
			if ((maxVersion())[1] < 19000) do
				windows.processpostedmessages()
			
			m_display_corruptionFoundInEnv_dialog_return_value = false-- default return value
			autobackup_enabled = autobackup.enabled  -- disable autobackup while dialog is displayed
			autobackup.enabled = false
		)
		on m_corruptionFoundInEnv_ro close do
		(
			m_corruptionFoundInEnv_ro.height = m_orig_height
			autobackup.enabled = autobackup_enabled
		)
		on b_learn pressed do
		(
			show_help_for_topic m_help_topic_id
		)
		on b_clean pressed do
		(
			m_display_corruptionFoundInEnv_dialog_return_value = true
			destroyDialog m_corruptionFoundInEnv_ro
		)
		on b_leave pressed do
		(
			m_display_corruptionFoundInEnv_dialog_return_value = false
			destroyDialog m_corruptionFoundInEnv_ro
		)
		on m_corruptionFoundInEnv_ro help do
		(
			show_help_for_topic m_help_topic_id
		)
	),

	-- display corruption found in env dialog.  Returns whether to clean the env.
	fn display_corruptionFoundInEnv_dialog contaminated_files help_topic_id which_script =
	(
		m_filelist = contaminated_files
		m_help_topic_id = help_topic_id
		m_which_script = which_script
		createdialog m_corruptionFoundInEnv_ro bgcolor:m_dialog_background_color modal:true escapeEnable:true
		m_display_corruptionFoundInEnv_dialog_return_value -- return value
	),

	-- the rollout definition for corruption cleaned in env
	m_corruptionCleanedInEnv_ro = rollout m_corruptionCleanedInEnv_ro ~CORRUPTION_CLEANED_TITLE~ width:~CORRUPTION_CLEANED_WIDTH~
	(
		local autobackup_enabled -- whether autobackup was enabled when opening the dialog
		imgtag icon width:32 height:32 align:#left style:#bmp_center applyUIScaling:false
		label l1 ~CORRUPT_SCRIPT_FILES_CLEANED~ offset:~MAIN_INSTRUCTION_OFFSET~ align:#left
		label l2 ""
		button b_close ~CLOSE~ align:#right
		
		on m_corruptionCleanedInEnv_ro open do
		(
			set_imgtag_bitmap icon m_success_icon_filename
			autobackup_enabled = autobackup.enabled  -- disable autobackup while dialog is displayed
			autobackup.enabled = false
		)
		on m_corruptionCleanedInEnv_ro close do
		(
			autobackup.enabled = autobackup_enabled
		)
		on b_close pressed do
		(
			destroyDialog m_corruptionCleanedInEnv_ro
		)
		on m_corruptionCleanedInEnv_ro help do
		(
			show_help_for_topic m_help_topic_id
		)
	),

	-- display corruption cleaned in env dialog
	fn display_corruptionCleanedInEnv_dialog help_topic_id =
	(
		m_help_topic_id = help_topic_id
		createdialog m_corruptionCleanedInEnv_ro bgcolor:m_dialog_background_color modal:true escapeEnable:true
	),

	-- the rollout definition for corruption not cleaned in env
	m_corruptionNotCleanedInEnv_ro = rollout m_corruptionNotCleanedInEnv_ro ~ENV_CORRUPTION_NOT_CLEANED_TITLE~ width:~ENV_CORRUPTION_NOT_CLEANED_WIDTH~
	(
		local autobackup_enabled -- whether autobackup was enabled when opening the dialog
		imgtag icon width:32 height:32 align:#left style:#bmp_center applyUIScaling:false
		label l1 ~ENV_CORRUPT_SCRIPTS_NOT_CLEANED~ offset:~MAIN_INSTRUCTION_OFFSET~ align:#left
		label l2 ""
		label l3 ~ENV_CORRUPTION_NOT_CLEANED_MSG~ height:~ENV_CORRUPTION_NOT_CLEANED_MSG_HEIGHT~ align:#left
		label l4 ""
		button b_learn ~LEARN~ across:2 align:#left
		button b_close ~CLOSE~ align:#right
		
		on m_corruptionNotCleanedInEnv_ro open do
		(
			set_imgtag_bitmap icon m_warning_icon_filename
			autobackup_enabled = autobackup.enabled  -- disable autobackup while dialog is displayed
			autobackup.enabled = false
		)
		on m_corruptionNotCleanedInEnv_ro close do
		(
			autobackup.enabled = autobackup_enabled
		)
		on b_learn pressed do
		(
			show_help_for_topic m_help_topic_id
		)
		on b_close pressed do
		(
			destroyDialog m_corruptionNotCleanedInEnv_ro
		)
		on m_corruptionNotCleanedInEnv_ro help do
		(
			show_help_for_topic m_help_topic_id
		)
	),

	-- display corruption not cleaned in env dialog
	fn display_corruptionNotCleanedInEnv_dialog help_topic_id =
	(
		m_help_topic_id = help_topic_id
		createdialog m_corruptionNotCleanedInEnv_ro bgcolor:m_dialog_background_color modal:true escapeEnable:true
	),

	-- the filelist count for the first time the m_corruptionErrorCleaningInEnv_ro is displayed. We are playing games with the et_filelist height and the layout of controls under it. Need
	-- this value to tweak the rollout size
	m_corruptionErrorCleaningInEnv_ro_first_filelist_count,
	
	-- the rollout definition for error cleaning corruption in env
	m_corruptionErrorCleaningInEnv_ro = rollout m_corruptionErrorCleaningInEnv_ro ~CORRUPTION_CLEAN_FAILED_IN_ENV_TITLE~ width:~CORRUPTION_CLEAN_FAILED_IN_ENV_WIDTH~
	(
		-- local initial values are evaluated at compile time. We use m_filelist_height as the height for et_filelist so that controls under it are laid out correctly and the rollout height is ok.
		local m_filelist_height = if SecurityToolsDialogs_instance.m_filelist == undefined then 16 else SecurityToolsDialogs_instance.m_filelist.count * 16
		local autobackup_enabled -- whether autobackup was enabled when opening the dialog
		local m_orig_height -- restore initial rollout height in close handler

		imgtag icon width:32 height:32 align:#left style:#bmp_center applyUIScaling:false
		label l1 ~CORRUPT_SCRIPTS_NOT_CLEANED~ offset:~MAIN_INSTRUCTION_OFFSET~ align:#left
		label l2 ""
		label l3 ~CLEAN_FAILED_FOR~ align:#left
		edittext et_filelist height:m_filelist_height readOnly:true
		label l4 ""
		button b_learn ~LEARN~ across:2 align:#left
		button b_leave ~CLOSE~ align:#right

		on m_corruptionErrorCleaningInEnv_ro open do
		(
			set_imgtag_bitmap icon m_error_icon_filename
			local content = ""
			local first_entry = true
			for filename in m_filelist do 
			(
				if not first_entry do content += "\n"
				first_entry = false
				content += filename
			)
			et_filelist.text = content

			-- futz with height of rollout. May be easier to not use m_filelist_height as param for et_filelist, but rather set the height in this open handler
			-- and manually reposition controls below it.
			m_orig_height = m_corruptionFoundInEnv_ro.height
			if m_corruptionErrorCleaningInEnv_ro_first_filelist_count == undefined then
				m_corruptionErrorCleaningInEnv_ro_first_filelist_count = m_filelist.count
			m_corruptionErrorCleaningInEnv_ro.height += (m_filelist.count - m_corruptionErrorCleaningInEnv_ro_first_filelist_count) * 16 
			-- for 3ds Max 2016 and earlier following is needed to get the dialog height to update when inside the open handler
			if ((maxVersion())[1] < 19000) do
				windows.processpostedmessages()

			autobackup_enabled = autobackup.enabled  -- disable autobackup while dialog is displayed
			autobackup.enabled = false
		)
		on m_corruptionErrorCleaningInEnv_ro close do
		(
			autobackup.enabled = autobackup_enabled
		)
		on b_learn pressed do
		(
			show_help_for_topic m_help_topic_id
		)
		on b_leave pressed do
		(
			destroyDialog m_corruptionErrorCleaningInEnv_ro
		)
		on m_corruptionErrorCleaningInEnv_ro help do
		(
			show_help_for_topic m_help_topic_id
		)
	),

	-- display error cleaning corruption in env dialog
	fn display_corruptionErrorCleaningInEnv_dialog uncleaned_files help_topic_id =
	(
		m_help_topic_id = help_topic_id
		m_filelist = uncleaned_files
		createdialog m_corruptionErrorCleaningInEnv_ro bgcolor:m_dialog_background_color modal:true escapeEnable:true
	),

	-- the display_security_tools_startup_permission_dialog function return value
	m_SecurityToolsStartup_ro_return_value,
	
	-- the state of security tools startup. Set by display_security_tools_startup_permission_dialog
	m_SecurityToolsStartup_state,
	
	-- the rollout definition for the security tools startup info / permission dialog
	m_SecurityToolsStartup_ro = rollout m_SecurityToolsStartup_ro ~SECURITY_TOOLS_STARTUP_RO_TITLE~ width:~SECURITY_TOOLS_STARTUP_RO_WIDTH~
	(
		imgtag icon width:32 height:32 align:#left style:#bmp_center applyUIScaling:false
		label l1 ~SECURITY_TOOLS_STARTUP_RO_L1_TEXT~ align:#left offset:~MAIN_INSTRUCTION_OFFSET~
		label l2 ""
		label l3 ~SECURITY_TOOLS_STARTUP_RO_L3_TEXT~ align:#left height:~SECURITY_TOOLS_STARTUP_RO_L3_HEIGHT~
		label l4 ""
		checkbox cb_enabled ~SECURITY_TOOLS_STARTUP_RO_CB_ENABLED_TEXT~
		label l7 ""
		button b_learn ~LEARN_MORE~ across:2 align:#left
		button b_close ~OK~ align:#right
		on b_close pressed do 
		(
			destroydialog m_SecurityToolsStartup_ro
		)
		on m_SecurityToolsStartup_ro open do
		(
			set_imgtag_bitmap icon m_info_icon_filename
			cb_enabled.checked = m_SecurityToolsStartup_state
		)
		on b_learn pressed do
		(
			show_help_for_topic m_help_topic_id
		)
		on m_SecurityToolsStartup_ro close do
		(
			if cb_enabled.checked then
			(
				m_SecurityToolsStartup_ro_return_value = true
			)
			else
			(
				m_SecurityToolsStartup_ro_return_value = false
			)
		)
		on m_SecurityToolsStartup_ro help do
		(
			show_help_for_topic m_help_topic_id
		)
	),
	
	-- display the security tools startupinfo / permission dialog as a modal dialog, set ini file to persist that user has seen the dialog
	-- returns true to load security tools, false to disable security tools
	fn display_security_tools_startup_permission_dialog security_tools_startup_state help_topic_id =
	(
		m_SecurityToolsStartup_ro_returnvalue = true
		m_SecurityToolsStartup_state = security_tools_startup_state
		m_help_topic_id = help_topic_id
		createdialog m_SecurityToolsStartup_ro bgcolor:m_dialog_background_color modal:true escapeEnable:true
		m_SecurityToolsStartup_ro_return_value -- return value
	),

	on create do
	(
		m_dialog_background_color = ((colorman.getcolor #window)*255) as color
		
		local security_tools_scripts_dir = pathconfig.removePathLeaf (getThisScriptFilename())
		local security_tools_icon_dir = pathconfig.appendPath (pathconfig.removePathLeaf security_tools_scripts_dir) "icons"
		local color_theme = "dark"
		if isproperty colorman #getColorTheme then 
			color_theme = colorman.getColorTheme() as string -- "light" or "dark"
		else
		(
			local icon_folder = colorman.getIconFolder() -- something like: "E:\Program Files\Autodesk\3ds Max 2016\UI_ln\IconsDark"
			if icon_folder.count >= 5 do
			(
				local maybe_light = substring icon_folder (icon_folder.count-4) 5 -- "Light" or "sDark"
				if (stricmp maybe_light "light") == 0 then
					color_theme = "light"
				else
					color_theme = "dark"
			)
		)
		security_tools_icon_dir = pathconfig.appendPath security_tools_icon_dir color_theme
		local dpi_scaling = 1.
		if (globalvars.isGlobal #CustomControlsOptions) do 
		(
			dpi_scaling = CustomControlsOptions.UIUserScaleFactor
			if dpi_scaling == 0. do
				dpi_scaling = GetUIScaleFactor()
		)
		-- supported scaling values: 1, 1.25, 1.5, 2.0
		local icon_size = case of 
		(
			(dpi_scaling < 1.2): "32"
			(dpi_scaling < 1.4): "40"
			(dpi_scaling < 1.9): "48"
			default: "64"
		)

		m_error_icon_filename = pathconfig.appendPath security_tools_icon_dir ("error_" + icon_size + ".png")
		m_info_icon_filename = pathconfig.appendPath security_tools_icon_dir ("info_" + icon_size + ".png")
		m_success_icon_filename = pathconfig.appendPath security_tools_icon_dir ("success_" + icon_size + ".png")
		m_warning_icon_filename = pathconfig.appendPath security_tools_icon_dir ("warning_" + icon_size + ".png")
		
		-- by default, use help file in package bundle if less than 3ds Max 2019, otherwise use online help file
		m_help_use_bundle_help = (maxVersion())[1] < 21000
		
		local help_base_dir = pathconfig.appendPath (pathconfig.removePathLeaf security_tools_scripts_dir) "Help"
		local lang = (sysinfo.GetMaxLanguage())[5] -- for example, "en-US"
		local help_dir = pathconfig.appendPath help_base_dir lang
		local help_file = pathconfig.appendPath help_dir m_help_bundle_filename
		if (doesFileExist help_file) then
			m_help_filename = help_file
		else
		(
			help_dir = pathconfig.appendPath help_base_dir "en-US" -- use "en-US" as a defaule
			help_file = pathconfig.appendPath help_dir m_help_bundle_filename
			if (doesFileExist help_file) do
				m_help_filename = help_file
		)
		
		try
		(
			local maxPlusAssembly = dotNet.loadAssembly ((symbolicPaths.getPathValue "$max") + @"\MaxPlusDotNet.dll")
			local MaxPlus_Application = dotnetclass "Autodesk.Max.MaxPlus.Application"
			m_enableDrawing_method = MaxPlus_Application.EnableDrawing
		)
		catch ()
	)
)
::SecurityToolsDialogs_instance = SecurityToolsDialogs_structdef()


-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQCappIv1W+ViwavaPlhdTnIyrDlAVfpFMD4
-- KWNjLAP4HHHjLQOiYECJp36i637tjNQxailmMnktZGy1NncCU1kx+W2iMaIbona8
-- 5SxU4RsoWfMoDOHixO/sOT8skkqd6wm+5gPq4xnnRQMFFBS/Itrj2MGTUXxr5en5
-- kzTUgDiwzFSuXhtEWqDMr1sQLcP+KF+uJecgyEl9r4CQkMZ9U9FthcrN8PXrIr6Q
-- xGrC9FD6DXXWT026AqbdztznJQzXZP3ICAT00rDaceKXhlv87T/vuu5WfW6PNIWB
-- flx5LNxHEkiHFFBiy5sLF63WTKLIc1xyD8pjO58/EaItFdPN9gjy
-- -----END-SIGNATURE-----